/*
Title: Video Player
Author: DKPlugins
Site: https://dk-plugins.ru
E-mail: kuznetsovdenis96@gmail.com
Version: 1.4.2
Release: 31.05.2023
First release: 17.02.2021
*/


/*:
 * @plugindesc v.1.4.2 [PRO] [MV|MZ] Adds video to the title screen, credits, the layers on the map and other.
 * @author DKPlugins
 * @url https://dk-plugins.ru
 * @target MZ
 * @orderAfter DKTools
 * @orderAfter DK_Game_Time
 * @help

 ### Info about plugin ###
 Title: DK_Video_Player
 Author: DKPlugins
 Site: https://dk-plugins.ru
 Version: 1.4.2
 Release: 31.05.2023
 First release: 17.02.2021

 ###===========================================================================
 ## Requirements and dependencies
 ###===========================================================================
 GIF playback requires availability of working plugin pixi-gif
 Download page: https://dk-plugins.ru/pixi-gif/

 ###=========================================================================
 ## Compatibility
 ###=========================================================================
 RPG Maker MV: 1.5+
 RPG Maker MZ: 1.0+

 ###=========================================================================
 ## Compatibility with other plugins
 ###=========================================================================
 For compatibility to work properly,
 plugins from the following list must be placed ABOVE:
 DKTools.js
 DK_Game_Time.js

 ###=========================================================================
 ## Instructions
 ###=========================================================================
 Before calling the PlayVideo plugin command, you must call the LoadVideo or
 LoadGIF command to load the video/GIF onto the layer (preload to memory)!

 ## Using video as map parallax ##
 In the map notes, write <videoParallax: filename>
 filename - Video filename without extension
 Example: <videoParallax: map_parallax_video>

 For parallax looping, use the built-in looping settings in the map options

 ###=========================================================================
 ## Plugin commands (RPG Maker MV)
 ###=========================================================================
 1. Load video onto layer: LoadVideo LAYER SRC
 LAYER - Layer. Calculated with Javascript.
 SRC - Video source. Without "movies/" and file extension.
 Example: LoadVideo 1 dialog1
 Example: LoadVideo $gameVariables.value(1) dialog1

 2. Play video (must be pre-loaded on the layer): PlayVideo LAYER LOOP WAIT
 LAYER - Layer. Calculated with Javascript.
 LOOP - Loop video (true, false)
 WAIT - Wait for playing (true, false)
 Example: PlayVideo 1 true false
 Example: PlayVideo $gameVariables.value(1) false true

 3. Pause video: PauseVideo LAYER
 LAYER - Layer. Calculated with Javascript.
 Example: PauseVideo 1
 Example: PauseVideo $gameVariables.value(1)

 4. Resume video: ResumeVideo LAYER
 LAYER - Layer. Calculated with Javascript.
 Example: ResumeVideo 1
 Example: ResumeVideo $gameVariables.value(1)

 5. Stop video: StopVideo LAYER
 LAYER - Layer. Calculated with Javascript.
 Example: StopVideo 1
 Example: StopVideo $gameVariables.value(1)

 6. Replace video on the layer*: ReplaceVideo LAYER SRC LOOP WAIT
 *note: The change occurs after a new video is loaded and the previous video ends.
 LAYER - Layer. Calculated with Javascript.
 SRC - Video source. Without "movies/" and file extension.
 LOOP - Loop video (true, false)
 WAIT - Wait for playing (true, false)
 Example: ReplaceVideo 1 dialog1 true false
 Example: ReplaceVideo $gameVariables.value(1) dialog1 true false

 7. Set video volume: SetVideoVolume LAYER VOLUME
 LAYER - Layer. Calculated with Javascript.
 VOLUME - Volume. Calculated with Javascript.
 Example: SetVideoVolume 1 100
 Example: SetVideoVolume $gameVariables.value(1) $gameVariables.value(2)

 8. Set video speed: SetVideoSpeed LAYER SPEED
 LAYER - Layer. Calculated with Javascript.
 SPEED - Video speed. Calculated with Javascript.
 Example: SetVideoSpeed 1 150
 Example: SetVideoSpeed $gameVariables.value(1) $gameVariables.value(2)

 9. Set video blend mode: SetVideoBlendMode LAYER BLEND_MODE
 LAYER - Layer. Calculated with Javascript.
 BLEND_MODE - Blend mode (NORMAL, ADD, MULTIPLY, SCREEN)
 Example: SetVideoBlendMode 1 ADD
 Example: SetVideoBlendMode $gameVariables.value(1) MULTIPLY

 10. Set video pivot: SetVideoPivot LAYER PIVOT
 LAYER - Layer. Calculated with Javascript.
 PIVOT - Pivot (0 - upper left, 0.5 - center)
 Example: SetVideoPivot 1 0.5
 Example: SetVideoPivot $gameVariables.value(1) $gameVariables.value(2)

 11. Move video: MoveVideo LAYER X Y DURATION
 LAYER - Layer. Calculated with Javascript.
 X - X. Calculated with Javascript.
 Y - Y. Calculated with Javascript.
 DURATION - Duration in frames. 0 - Immediately. Calculated with Javascript.
 Example: MoveVideo 1 100 100 0
 Example: MoveVideo $gameVariables.value(1) $gameVariables.value(2) $gameVariables.value(3) $gameVariables.value(4)

 12. Change video opacity: ChangeVideoOpacity LAYER OPACITY DURATION
 LAYER - Layer. Calculated with Javascript.
 OPACITY - Opacity. Calculated with Javascript.
 DURATION - Duration in frames. 0 - Immediately. Calculated with Javascript.
 Example: ChangeVideoOpacity 1 128 60
 Example: ChangeVideoOpacity $gameVariables.value(1) $gameVariables.value(2) $gameVariables.value(3)

 13. Change video scale: ChangeVideoScale LAYER X Y DURATION
 LAYER - Layer. Calculated with Javascript.
 X - Scale X. Calculated with Javascript.
 Y - Scale Y. Calculated with Javascript.
 DURATION - Duration in frames. 0 - Immediately. Calculated with Javascript.
 Example: ChangeVideoScale 1 50 50 60
 Example: ChangeVideoScale $gameVariables.value(1) $gameVariables.value(2) $gameVariables.value(3) $gameVariables.value(4)

 14. Stop all video: StopAllVideo

 15. Place the video player in front of the images layer: PlaceVideoPlayerBeforePictures

 16. Place video player after images layer: PlaceVideoPlayerAfterPictures

 17. Reset layer (position, opacity, volume, etc.): ResetVideoLayer LAYER
 LAYER - Layer. Calculated with Javascript.
 Example: ResetVideoLayer 1
 Example: ResetVideoLayer $gameVariables.value(1)

 18. Wait for all videos to load: WaitVideoPlayerLoading

 ###=========================================================================
 ## See also
 ###=========================================================================
 1. Game Time (https://dk-plugins.ru/game-time/)
 Time system.

 2. Globals (https://dk-plugins.ru/globals/)
 Allows you to specify variables and switches that will be â€œglobalâ€ to all player saves.
 Changes are saved in a separate file and applied when starting a new game or loading any save.

 3. Mouse System (https://dk-plugins.ru/mouse-system/)
 Allows you to change the mouse cursor, activate events by clicking, hovering, etc.

 4. Events Glow (https://dk-plugins.ru/events-glow/)
 Allows highlighting events on mouse hover.

 5. Pictures Glow (https://dk-plugins.ru/pictures-glow/)
 Allows highlighting pictures on mouse hover.

 ###=========================================================================
 ## Graphics
 ###=========================================================================
 Additional graphics for your project: https://dk-plugins.ru/resources/

 ###=========================================================================
 ## License and terms of use
 ###=========================================================================
 You can:
 -To use the plugin for your non-commercial projects
 -Change code of the plugin

 You cannot:
 -Delete or change any information about the plugin
 -Distribute the plugin and its modifications

 ## Commercial license ##
 Visit the page: https://dk-plugins.ru/commercial-license/

 ###=========================================================================
 ## Support
 ###=========================================================================
 Become a subscriber: https://boosty.to/dkplugins



 * @command LoadVideo
 * @text Load Video
 * @desc Loads the video onto the layer
 *
 * @arg layer
 * @text Layer
 * @desc Layer. Calculated with Javascript.
 * @type combo
 * @option 1
 * @option $gameVariables.value(ID)
 * @default 1
 *
 * @arg src
 * @text Video source
 * @desc Video source. Without "movies/" and file extension.

 * @command LoadGIF
 * @text Load GIF
 * @desc Loads the GIF onto the layer
 *
 * @arg layer
 * @text Layer
 * @desc Layer. Calculated with Javascript.
 * @type combo
 * @option 1
 * @option $gameVariables.value(ID)
 * @default 1
 *
 * @arg src
 * @text GIF source
 * @desc GIF source. Without "movies/" and file extension.

 * @command PlayVideo
 * @text Play Video/GIF
 * @desc Plays the video/GIF (needs to be pre-loaded on layer)
 *
 * @arg layer
 * @text Layer
 * @desc Layer. Calculated with Javascript.
 * @type combo
 * @option 1
 * @option $gameVariables.value(ID)
 * @default 1
 *
 * @arg loop
 * @text Loop
 * @desc Loop video/GIF ?
 * @type boolean
 * @default false
 *
 * @arg wait
 * @text Wait
 * @desc Wait for playing
 * @type boolean
 * @default false

 * @command PauseVideo
 * @text Pause Video/GIF
 * @desc Pauses the video/GIF
 *
 * @arg layer
 * @text Layer
 * @desc Layer. Calculated with Javascript.
 * @type combo
 * @option 1
 * @option $gameVariables.value(ID)
 * @default 1

 * @command ResumeVideo
 * @text Resume Video/GIF
 * @desc Resumes the video/GIF
 *
 * @arg layer
 * @text Layer
 * @desc Layer. Calculated with Javascript.
 * @type combo
 * @option 1
 * @option $gameVariables.value(ID)
 * @default 1

 * @command StopVideo
 * @text Stop video/GIF
 * @desc Stops the video/GIF
 *
 * @arg layer
 * @text Layer
 * @desc Layer. Calculated with Javascript.
 * @type combo
 * @option 1
 * @option $gameVariables.value(ID)
 * @default 1

 * @command ReplaceVideo
 * @text Replace Video
 * @desc Replaces the video/GIF with video. The video changes at the end of the previous one.
 *
 * @arg layer
 * @text Layer
 * @desc Layer. Calculated with Javascript.
 * @type combo
 * @option 1
 * @option $gameVariables.value(ID)
 * @default 1
 *
 * @arg src
 * @text Video source
 * @desc Video source. Without "movies/" and file extension.
 *
 * @arg loop
 * @text Loop
 * @desc Loop video ?
 * @type boolean
 * @default false
 *
 * @arg wait
 * @text Wait
 * @desc Wait for playing
 * @type boolean
 * @default false

 * @command ReplaceGIF
 * @text Replace GIF
 * @desc Replaces the video/GIF with GIF. The GIF changes at the end of the previous one.
 *
 * @arg layer
 * @text Layer
 * @desc Layer. Calculated with Javascript.
 * @type combo
 * @option 1
 * @option $gameVariables.value(ID)
 * @default 1
 *
 * @arg src
 * @text GIF source
 * @desc GIF source. Without "movies/" and file extension.
 *
 * @arg loop
 * @text Loop
 * @desc Loop GIF ?
 * @type boolean
 * @default false
 *
 * @arg wait
 * @text Wait
 * @desc Wait for playing
 * @type boolean
 * @default false

 * @command WaitVideoPlayerLoading
 * @desc Waiting for all videos to load

 * @command SetVideoVolume
 * @desc Sets the video volume
 *
 * @arg layer
 * @text Layer
 * @desc Layer. Calculated with Javascript.
 * @type combo
 * @option 1
 * @option $gameVariables.value(ID)
 * @default 1
 *
 * @arg volume
 * @text Volume
 * @desc Volume. Default: 100
 * @default 100

 * @command SetVideoSpeed
 * @desc Sets the video speed
 *
 * @arg layer
 * @text Layer
 * @desc Layer. Calculated with Javascript.
 * @type combo
 * @option 1
 * @option $gameVariables.value(ID)
 * @default 1
 *
 * @arg speed
 * @text Speed
 * @desc Speed. Default: 100
 * @default 100

 * @command SetVideoBlendMode
 * @desc Sets the video blend mode
 *
 * @arg layer
 * @text Layer
 * @desc Layer. Calculated with Javascript.
 * @type combo
 * @option 1
 * @option $gameVariables.value(ID)
 * @default 1
 *
 * @arg blendMode
 * @text Blend mode
 * @desc Blend mode
 * @type select
 * @option normal
 * @value NORMAL
 * @option add
 * @value ADD
 * @option multiply
 * @value MULTIPLY
 * @option screen
 * @value SCREEN
 * @default NORMAL

 * @command SetVideoPivot
 * @desc Sets the video pivot
 *
 * @arg layer
 * @text Layer
 * @desc Layer. Calculated with Javascript.
 * @type combo
 * @option 1
 * @option $gameVariables.value(ID)
 * @default 1
 *
 * @arg pivot
 * @text Pivot
 * @desc Pivot
 * @type select
 * @option Upper left
 * @value 0
 * @option Center
 * @value 0.5
 * @default 0

 * @command MoveVideo
 * @text Move Video/GIF
 * @desc Moves the video/GIF
 *
 * @arg layer
 * @text Layer
 * @desc Layer. Calculated with Javascript.
 * @type combo
 * @option 1
 * @option $gameVariables.value(ID)
 * @default 1
 *
 * @arg x
 * @text X
 * @desc X. Calculated with Javascript.
 * @type combo
 * @option 0
 * @option $gameVariables.value(ID)
 * @default 0
 *
 * @arg y
 * @text Y
 * @desc Y. Calculated with Javascript.
 * @type combo
 * @option 0
 * @option $gameVariables.value(ID)
 * @default 0
 *
 * @arg duration
 * @text Duration
 * @desc Duration in frames. 0 - Immediately. Calculated with Javascript.
 * @type combo
 * @option 0
 * @option $gameVariables.value(ID)
 * @default 0

 * @command ChangeVideoOpacity
 * @desc Changes the video opacity
 *
 * @arg layer
 * @text Layer
 * @desc Layer. Calculated with Javascript.
 * @type combo
 * @option 1
 * @option $gameVariables.value(ID)
 * @default 1
 *
 * @arg opacity
 * @text Opacity
 * @desc Opacity. Calculated with Javascript.
 * @type combo
 * @option 255
 * @option 0
 * @option $gameVariables.value(ID)
 * @default 255
 *
 * @arg duration
 * @text Duration
 * @desc Duration in frames. 0 - Immediately. Calculated with Javascript.
 * @type combo
 * @option 0
 * @option $gameVariables.value(ID)
 * @default 0

 * @command ChangeVideoScale
 * @desc Changes the video scale
 *
 * @arg layer
 * @text Layer
 * @desc Layer. Calculated with Javascript.
 * @type combo
 * @option 1
 * @option $gameVariables.value(ID)
 * @default 1
 *
 * @arg x
 * @text X
 * @desc X. Calculated with Javascript.
 * @type combo
 * @option 100
 * @option 50
 * @option 150
 * @option $gameVariables.value(ID)
 * @default 100
 *
 * @arg y
 * @text Y
 * @desc Y. Calculated with Javascript.
 * @type combo
 * @option 100
 * @option 50
 * @option 150
 * @option $gameVariables.value(ID)
 * @default 100
 *
 * @arg duration
 * @text Duration
 * @desc Duration in frames. 0 - Immediately. Calculated with Javascript.
 * @type combo
 * @option 0
 * @option $gameVariables.value(ID)
 * @default 0

 * @command StopAllVideo
 * @desc Stops all videos

 * @command PlaceVideoPlayerBeforePictures
 * @desc Place the video player in front of the images layer

 * @command PlaceVideoPlayerAfterPictures
 * @desc Place video player after images layer

 * @command ResetVideoLayer
 * @desc Reset layer (position, opacity, volume, etc.)
 *
 * @arg layer
 * @text Layer
 * @desc Layer. Calculated with Javascript.
 * @type combo
 * @option 1
 * @option $gameVariables.value(ID)
 * @default 1

 * @param videoExtension
 * @text Video extension
 * @desc Video extension
 * @type combo
 * @option auto
 * @option .webm
 * @option .mp4
 * @default auto




 * @param Title screen
 * @default ---------------------------------

 * @param titleVideos
 * @text Videos before title screen
 * @parent Title screen
 * @desc Videos before title screen
 * @type struct<TitleVideo>
 * @default {"videos":"[]","okBehavior":"","cancelBehavior":"","playbackRate":"150"}

 * @param titleBackground
 * @text Background video
 * @parent Title screen
 * @desc Background video on title screen
 * @type struct<Video>
 * @default {"src":"","volume":"100","playbackRate":"100"}

 * @param creditsVideos
 * @text Credits videos
 * @parent Title screen
 * @desc Credits videos on title screen
 * @type struct<CreditsVideo>
 * @default {"commandName":"","videos":"[]","okBehavior":"","cancelBehavior":"","playbackRate":"150"}

 * @param Map
 * @default ---------------------------------

 * @param mapMaxLayers
 * @text Maximum layers on map
 * @parent Map
 * @desc Maximum layers on map
 * @type number
 * @min 1
 * @default 5

 * @param mapOkBehavior
 * @text "ok" behavior
 * @parent Map
 * @desc Behavior for "ok" (enter, mouse left button, etc.). Only works with looped videos.
 * @type select
 * @option None
 * @value
 * @option Skip
 * @value skip
 * @default

 * @param mapCancelBehavior
 * @text "cancel" behavior
 * @parent Map
 * @desc Behavior for "cancel" (escape, mouse right button, etc.). Only works with looped videos.
 * @type select
 * @option None
 * @value
 * @option Skip
 * @value skip
 * @default

 * @param Battle
 * @default ---------------------------------

 * @param statusWindowAutoHide
 * @text Automatic hiding of the status window
 * @parent Battle
 * @desc Hide status window when video is playing?
 * @type boolean
 * @default false

 * @param battleMaxLayers
 * @text Maximum layers on the map
 * @parent Battle
 * @desc Maximum layers on the map
 * @type number
 * @min 1
 * @default 5

 * @param battleOkBehavior
 * @text "ok" behavior
 * @parent Battle
 * @desc Behavior for "ok" (enter, mouse left button, etc.). Only works with looped videos.
 * @type select
 * @option None
 * @value
 * @option Skip
 * @value skip
 * @default

 * @param battleCancelBehavior
 * @text "cancel" behavior
 * @parent Battle
 * @desc Behavior for "cancel" (escape, mouse right button, etc.). Only works with looped videos.
 * @type select
 * @option None
 * @value
 * @option Skip
 * @value skip
 * @default

 * @param Menu
 * @default ---------------------------------

 * @param menuBackground
 * @text Menu background
 * @parent Menu
 * @desc Video source of the menu background. Without "movies/" and file extension.

 * @param RPG Maker MV
 * @default ---------------------------------

 * @param useMasterVolume
 * @text Use master volume
 * @parent RPG Maker MV
 * @desc Use master volume for videos
 * @type boolean
 * @default false

*/



'use strict';

var Imported = Imported || {};
Imported['DK_Video_Player'] = '1.4.2';

//===========================================================================
// initialize parameters
//===========================================================================

const VideoPlayerParams = (function() {

    function parse(string) {
        try {
            return JSON.parse(string, function(key, value) {
                if (typeof string === 'number' || typeof string === 'boolean') {
                    return string;
                }

                try {
                    if (Array.isArray(value)) {
                        return value.map(val => parse(val));
                    }

                    return parse(value);
                } catch (e) {
                    return value;
                }
            });
        } catch(e) {
            return string;
        }
    }

    const parameters = PluginManager.parameters('DK_Video_Player');

    return Object.entries(parameters).reduce((acc, [key, value]) => {
        acc[key] = parse(value);

        return acc;
    }, {});

})();

//===========================================================================
// Game_Interpreter
//===========================================================================

const VideoPlayer_Game_Interpreter_pluginCommand =
    Game_Interpreter.prototype.pluginCommand;
Game_Interpreter.prototype.pluginCommand = function(command, args) {
    VideoPlayer_Game_Interpreter_pluginCommand.apply(this, arguments);

    switch (command) {
        case 'LoadVideo': {
            VideoManager.loadVideo(eval(args[0]), args[1]);
            break;
        }

        case 'SetVideoVolume': {
            VideoManager.setVideoVolume(eval(args[0]), eval(args[1]));
            break;
        }

        case 'SetVideoSpeed': {
            VideoManager.setVideoSpeed(eval(args[0]), eval(args[1]));
            break;
        }

        case 'SetVideoBlendMode': {
            VideoManager.setVideoBlendMode(eval(args[0]), PIXI.BLEND_MODES[args[1].toUpperCase()]);
            break;
        }

        case 'SetVideoPivot': {
            VideoManager.setVideoPivot(eval(args[0]), eval(args[1]));
            break;
        }

        case 'MoveVideo': {
            VideoManager.moveVideo(
                eval(args[0]),
                new Point(eval(args[1]), eval(args[2])),
                eval(args[3]));
            break;
        }

        case 'ChangeVideoOpacity': {
            VideoManager.changeVideoOpacity(eval(args[0]), eval(args[1]), eval(args[2]));
            break;
        }

        case 'ChangeVideoScale': {
            VideoManager.changeVideoScale(
                eval(args[0]),
                new Point(eval(args[1]) / 100, eval(args[2]) / 100),
                eval(args[3]))
            break;
        }

        case 'PlayVideo': {
            const layer = eval(args[0]);
            const options = {
                loop: args.length > 1 && (args[1].toLowerCase() === 'loop' || JSON.parse(args[1].toLowerCase()))
            };

            VideoManager.playVideo(layer, options);

            if (args.length > 2 && (args[2].toLowerCase() === 'wait' || JSON.parse(args[2].toLowerCase()))) {
                this.setWaitMode('videoPlayer-' + layer);
            }

            break;
        }

        case 'PauseVideo': {
            VideoManager.pauseVideo(eval(args[0]));
            break;
        }

        case 'ResumeVideo': {
            VideoManager.resumeVideo(eval(args[0]));
            break;
        }

        case 'StopVideo': {
            VideoManager.stopVideo(eval(args[0]));
            break;
        }

        case 'StopAllVideo': {
            VideoManager.stopAllVideo();
            break;
        }

        case 'ReplaceVideo': {
            const layer = eval(args[0]);
            const src = args[1];
            const options = {
                loop: args.length > 2 && (args[2].toLowerCase() === 'loop' || JSON.parse(args[2].toLowerCase()))
            };

            VideoManager.replaceVideo(layer, src, options);

            if (args.length > 3 && (args[3].toLowerCase() === 'wait' || JSON.parse(args[3].toLowerCase()))) {
                this.setWaitMode('videoPlayer-' + layer);
            }

            break;
        }

        case 'PlaceVideoPlayerBeforePictures': {
            const scene = SceneManager._scene;

            if (scene instanceof Scene_Map || scene instanceof Scene_Battle) {
                scene._spriteset && scene._spriteset.placeVideoPlayerBeforePictures();
            }
            break;
        }

        case 'PlaceVideoPlayerAfterPictures': {
            const scene = SceneManager._scene;

            if (scene instanceof Scene_Map || scene instanceof Scene_Battle) {
                scene._spriteset && scene._spriteset.placeVideoPlayerAfterPictures();
            }
            break;
        }

        case 'ResetVideoLayer': {
            VideoManager.resetLayer(eval(args[0]));
            break;
        }

        case 'WaitVideoPlayerLoading':
            this.setWaitMode('videoPlayerLoading');
            break;
    }
};

if (Utils.RPGMAKER_NAME === 'MZ') {

    PluginManager.registerCommand('DK_Video_Player', 'LoadVideo', (args) => {
        VideoManager.loadVideo(eval(args.layer), args.src);
    });

    PluginManager.registerCommand('DK_Video_Player', 'LoadGIF', (args) => {
        VideoManager.loadGIF(eval(args.layer), args.src);
    });

    PluginManager.registerCommand('DK_Video_Player', 'SetVideoVolume', (args) => {
        VideoManager.setVideoVolume(eval(args.layer), eval(args.volume));
    });

    PluginManager.registerCommand('DK_Video_Player', 'SetVideoSpeed', (args) => {
        VideoManager.setVideoSpeed(eval(args.layer), eval(args.speed));
    });

    PluginManager.registerCommand('DK_Video_Player', 'SetVideoBlendMode', (args) => {
        VideoManager.setVideoBlendMode(eval(args.layer), PIXI.BLEND_MODES[args.blendMode]);
    });

    PluginManager.registerCommand('DK_Video_Player', 'SetVideoPivot', (args) => {
        VideoManager.setVideoPivot(eval(args.layer), eval(args.pivot));
    });

    PluginManager.registerCommand('DK_Video_Player', 'MoveVideo', (args) => {
        VideoManager.moveVideo(
            eval(args.layer),
            new Point(eval(args.x), eval(args.y)),
            eval(args.duration));
    });

    PluginManager.registerCommand('DK_Video_Player', 'ChangeVideoOpacity', (args) => {
        VideoManager.changeVideoOpacity(
            eval(args.layer),
            eval(args.opacity),
            eval(args.duration));
    });

    PluginManager.registerCommand('DK_Video_Player', 'ChangeVideoScale', (args) => {
        VideoManager.changeVideoScale(
            eval(args.layer),
            new Point(eval(args.x) / 100, eval(args.y) / 100),
            eval(args.duration));
    });

    PluginManager.registerCommand('DK_Video_Player', 'PlayVideo', function(args) {
        const layer = eval(args.layer);
        const options = {
            loop: JSON.parse(args.loop)
        };

        VideoManager.playVideo(layer, options);

        if (JSON.parse(args.wait)) {
            this.setWaitMode('videoPlayer-' + layer);
        }
    });

    PluginManager.registerCommand('DK_Video_Player', 'PauseVideo', (args) => {
        VideoManager.pauseVideo(eval(args.layer));
    });

    PluginManager.registerCommand('DK_Video_Player', 'ResumeVideo', (args) => {
        VideoManager.resumeVideo(eval(args.layer));
    });

    PluginManager.registerCommand('DK_Video_Player', 'StopVideo', (args) => {
        VideoManager.stopVideo(eval(args.layer));
    });

    PluginManager.registerCommand('DK_Video_Player', 'StopAllVideo', () => {
        VideoManager.stopAllVideo();
    });

    PluginManager.registerCommand('DK_Video_Player', 'ReplaceVideo', function(args) {
        const layer = eval(args.layer);
        const src = args.src;
        const options = {
            loop: JSON.parse(args.loop)
        };

        VideoManager.replaceVideo(layer, src, options);

        if (JSON.parse(args.wait)) {
            this.setWaitMode('videoPlayer-' + layer);
        }
    });

    PluginManager.registerCommand('DK_Video_Player', 'ReplaceGIF', function(args) {
        const layer = eval(args.layer);
        const src = args.src;
        const options = {
            loop: JSON.parse(args.loop)
        };

        VideoManager.replaceGIF(layer, src, options);

        if (JSON.parse(args.wait)) {
            this.setWaitMode('videoPlayer-' + layer);
        }
    });

    PluginManager.registerCommand('DK_Video_Player', 'PlaceVideoPlayerBeforePictures', () => {
        const scene = SceneManager._scene;

        if (scene instanceof Scene_Map || scene instanceof Scene_Battle) {
            scene._spriteset && scene._spriteset.placeVideoPlayerBeforePictures();
        }
    });

    PluginManager.registerCommand('DK_Video_Player', 'PlaceVideoPlayerAfterPictures', () => {
        const scene = SceneManager._scene;

        if (scene instanceof Scene_Map || scene instanceof Scene_Battle) {
            scene._spriteset && scene._spriteset.placeVideoPlayerAfterPictures();
        }
    });

    PluginManager.registerCommand('DK_Video_Player', 'ResetVideoLayer', (args) => {
        VideoManager.resetLayer(eval(args.layer));
    });

    PluginManager.registerCommand('DK_Video_Player', 'WaitVideoPlayerLoading', function() {
        this.setWaitMode('videoPlayerLoading');
    });

}

const VideoPlayer_Game_Interpreter_updateWaitMode =
    Game_Interpreter.prototype.updateWaitMode;
Game_Interpreter.prototype.updateWaitMode = function() {
    if (this._waitMode === 'videoPlayerLoading') {
        const waiting = !VideoManager.isReady();

        if (!waiting) {
            this._waitMode = '';
        }

        return waiting;
    } else if (this._waitMode.startsWith('videoPlayer-')) {
        const layer = Number(this._waitMode.split('-').pop());
        const waiting = VideoManager.isPlaying(layer);

        if (!waiting) {
            this._waitMode = '';
        }

        return waiting;
    }

    return VideoPlayer_Game_Interpreter_updateWaitMode.apply(this, arguments);
};

//===========================================================================
// Game_Map
//===========================================================================

Game_Map.prototype.videoParallaxName = function() {
    if (!$dataMap.meta || !$dataMap.meta.videoParallax) {
        return null;
    }

    return $dataMap.meta.videoParallax.trim();
};

Game_Map.prototype.isVideoParallax = function() {
    return !!this.videoParallaxName();
};

//===========================================================================
// VideoManager
//===========================================================================

class VideoManager {

    /**
     * @static
     * @param {String} src
     * @return {String}
     */
    static getFullPath(src) {
        if (Imported['DKTools_Localization']) {
            const parts = (this.folder + '/' + src).split('/');
            const filename = parts.pop();
            let extension = this.videoExt();

            if (filename.endsWith('.webm') || filename.endsWith('.mp4') || filename.endsWith('.gif')) {
                extension = '';
            }

            const folder = DKTools.Localization.getVideoFolder(parts.join('/'), filename + extension);

            return folder + filename + extension;
        }

        if (src.endsWith('.webm') || src.endsWith('.mp4') || src.endsWith('.gif')) {
            return this.folder + src;
        }

        return this.folder + src + this.videoExt();
    }

    /**
     * @static
     * @param {String} src
     * @return {VideoElement | GifElement}
     */
    static load(src) {
        const fullPath = this.getFullPath(src);
        const videoElement = fullPath.endsWith('.gif') ?
            GifElement.load(fullPath) : VideoElement.load(fullPath);

        this._loadingQueue.push(videoElement);

        return videoElement;
    }

    /**
     * @static
     * @param {String} src
     * @return {Promise<VideoElement | GifElement>}
     */
    static loadAsync(src) {
        return new Promise((resolve) => {
            const videoElement = this.load(src);

            videoElement.addLoadListener(() => {
                resolve(videoElement);
            });
        });
    }

    /**
     * @static
     * @return {String}
     */
    static videoExt() {
        return VideoPlayerParams.videoExtension !== 'auto' ?
            VideoPlayerParams.videoExtension : Game_Interpreter.prototype.videoFileExt();
    }

    /**
     * @static
     * @return {Boolean}
     */
    static isReady() {
        if (this._loadingQueue.length > 0) {
            this._loadingQueue = this._loadingQueue.filter(el => !el.isReady());

            return this._loadingQueue.length === 0;
        }

        return true;
    }

    /**
     * @static
     * @param {Number} [layer]
     * @return {Boolean}
     */
    static isPlaying(layer) {
        return Boolean(this._videoPlayer && this._videoPlayer.isPlaying(layer))
            || this.isReplacing(layer);
    }

    /**
     * @static
     * @param {Number} [layer]
     * @return {Boolean}
     */
    static isReplacing(layer) {
        return Boolean(this._videoPlayer && this._videoPlayer.isReplacing(layer));
    }

    /**
     * @static
     * @return {Boolean}
     */
    static isBusy() {
        return !this.isReady() || this.isPlaying();
    }

    /**
     * @static
     * @param {Sprite_VideoPlayer} videoPlayer
     */
    static setVideoPlayer(videoPlayer) {
        this._videoPlayer = videoPlayer;
    }

    /**
     * @static
     * @param {Number} layer
     * @param {String} src
     */
    static loadVideo(layer, src) {
        this._videoPlayer && this._videoPlayer.load(layer, src);
    }

    /**
     * @static
     * @param {Number} layer
     * @param {String} src
     */
    static loadGIF(layer, src) {
        this._videoPlayer && this._videoPlayer.load(layer, src + '.gif');
    }

    /**
     * @static
     * @param {Number} layer
     * @return {Sprite_Video | null}
     */
    static getVideo(layer) {
        return (this._videoPlayer ?
            this._videoPlayer.getVideo(layer) : null);
    }

    /**
     * @static
     * @param {Number} layer
     * @param {Number} volume
     */
    static setVideoVolume(layer, volume) {
        this._videoPlayer && this._videoPlayer.setVolume(layer, volume);
    }

    /**
     * @static
     * @param {Number} layer
     * @param {Number} blendMode
     */
    static setVideoBlendMode(layer, blendMode) {
        this._videoPlayer && this._videoPlayer.setBlendMode(layer, blendMode);
    }

    /**
     * @static
     * @param {Number} layer
     * @param {Number} speed
     */
    static setVideoSpeed(layer, speed) {
        this._videoPlayer && this._videoPlayer.setSpeed(layer, speed);
    }

    /**
     * @static
     * @param {Number} layer
     * @param {Number} pivot
     */
    static setVideoPivot(layer, pivot) {
        this._videoPlayer && this._videoPlayer.setPivot(layer, pivot);
    }

    /**
     * @static
     * @param {Number} layer
     * @param {Point} pos
     * @param {Number} [duration=0]
     */
    static moveVideo(layer, pos, duration = 0) {
        this._videoPlayer && this._videoPlayer.changePos(layer, pos, duration);
    }

    /**
     * @static
     * @param {Number} layer
     * @param {Number} opacity
     * @param {Number} [duration=0]
     */
    static changeVideoOpacity(layer, opacity, duration = 0) {
        this._videoPlayer && this._videoPlayer.changeOpacity(layer, opacity, duration);
    }

    /**
     * @static
     * @param {Number} layer
     * @param {Point} scale
     * @param {Number} [duration=0]
     */
    static changeVideoScale(layer, scale, duration = 0) {
        this._videoPlayer && this._videoPlayer.changeScale(layer, scale, duration);
    }

    /**
     * @static
     * @param {Number} layer
     * @param {Object} [options={}]
     */
    static playVideo(layer, options = {}) {
        this._videoPlayer && this._videoPlayer.play(layer, options);
    }

    /**
     * @static
     * @param {Number} layer
     */
    static pauseVideo(layer) {
        this._videoPlayer && this._videoPlayer.pause(layer);
    }

    /**
     * @static
     * @param {Number} layer
     */
    static resumeVideo(layer) {
        this._videoPlayer && this._videoPlayer.resume(layer);
    }

    /**
     * @static
     * @param {Number} layer
     * @param {String} src
     * @param {Object} [options={}]
     */
    static replaceVideo(layer, src, options = {}) {
        this._videoPlayer && this._videoPlayer.replace(layer, src, options);
    }

    /**
     * @static
     * @param {Number} layer
     * @param {String} src
     * @param {Object} [options={}]
     */
    static replaceGIF(layer, src, options = {}) {
        this._videoPlayer && this._videoPlayer.replace(layer, src + '.gif', options);
    }

    /**
     * @static
     * @param {Number} layer
     */
    static resetLayer(layer) {
        this._videoPlayer && this._videoPlayer.resetLayer(layer);
    }

    /**
     * @static
     * @param {Number} layer
     */
    static stopVideo(layer) {
        this._videoPlayer && this._videoPlayer.stop(layer);
    }

    /**
     * @static
     */
    static stopAllVideo() {
        this._videoPlayer && this._videoPlayer.stopAll();
    }

    /**
     * @static
     */
    static update() {
        this._videoPlayer && this._videoPlayer.updateBehavior();
    }

    // static properties

    /**
     * @static
     * @return {String}
     */
    static get folder() {
        return 'movies/';
    }

}

VideoManager._loadingQueue = [];

//===========================================================================
// VideoElement
//===========================================================================

class VideoElement {

    constructor() {
        this.initialize.apply(this, arguments);
    }

    initialize(element) {
        this._loadListeners = [];
        this._endListeners = [];
        this._video = element;
        this._video.onloadeddata = this._onLoad.bind(this);
        this._video.onended = this._onEnd.bind(this);
    }

    // methods

    addLoadListener(listener) {
        if (this.isReady()) {
            listener(this);
        } else {
            this._loadListeners.push(listener);
        }
    }

    addEndListener(listener) {
        this._endListeners.push(listener);
    }

    isReady() {
        return !!this._isReady;
    }

    destroy() {
        this._video.src = '';
    }

    play(currentTime = 0) {
        this._video.currentTime = currentTime;
        this._video.play();
    }

    pause() {
        this._video.pause();
    }

    // private methods

    _onLoad() {
        this._isReady = true;

        while (this._loadListeners.length > 0) {
            this._loadListeners.shift()(this);
        }
    }

    _onEnd() {
        while (this._endListeners.length > 0) {
            this._endListeners.shift()(this);
        }
    }

    // properties

    get video() {
        return this._video;
    }

    get width() {
        return this._video.videoWidth;
    }

    get height() {
        return this._video.videoHeight;
    }

    get currentTime() {
        return this._video.currentTime;
    }

    get duration() {
        return this._video.duration;
    }

    // static methods

    static _createVideo(src) {
        const video = document.createElement('video');

        video.src = src;
        video.preload = 'auto';
        video.load();

        return video;
    }

    static load(src) {
        return new VideoElement(this._createVideo(src));
    }

}

//===========================================================================
// GifElement
//===========================================================================

class GifElement {

    constructor() {
        this.initialize.apply(this, arguments);
    }

    initialize(src) {
        this._loadListeners = [];
        this._endListeners = [];
        this._initializeVideo(src);
    }

    _initializeVideo(src) {
        this._video = new PIXI.Sprite();

        fetch(src).then((response) => response.arrayBuffer()).then((arrayBuffer) => {
            this._video = PIXI.gif.AnimatedGIF.fromBuffer(arrayBuffer, { autoPlay: false, loop: true });
            this._onLoad();
        });
    }

    addLoadListener(listener) {
        if (this.isReady()) {
            listener(this);
        } else {
            this._loadListeners.push(listener);
        }
    }

    addEndListener(listener) {
        debugger;
        this._endListeners.push(listener);
    }

    isReady() {
        return !!this._isReady;
    }

    destroy() {}

    play(currentTime = 0) {
        if (!this.isReady()) {
            return;
        }

        if (currentTime === 0) {
            this._video.currentFrame = 0;
        }

        this._video.play();
    }

    pause() {
        if (this.isReady()) {
            this._video.stop();
        }
    }

    updateFrame() {
        this._video.updateFrame();
    }

    // private methods

    _onLoad() {
        this._isReady = true;

        while (this._loadListeners.length > 0) {
            this._loadListeners.shift()(this);
        }
    }

    _onEnd() {
        while (this._endListeners.length > 0) {
            this._endListeners.shift()(this);
        }
    }

    // properties

    get video() {
        return this._video;
    }

    get width() {
        return this._video.width;
    }

    get height() {
        return this._video.height;
    }

    get currentTime() {
        return (this._video._currentTime || 0) / 1000;
    }

    get duration() {
        return (this._video.duration || 0) / 1000;
    }

    // static methods

    static load(src) {
        if (!PIXI.gif) {
            throw new Error('"DK_Video_Player" require "pixi-filters" plugin!');
        }

        return new GifElement(src);
    }

}

//===========================================================================
// Sprite_Video
//===========================================================================

class Sprite_Video extends Sprite {

    /**
     * @param {VideoElement} videoElement
     * @param {Object} [options={}]
     */
    initialize(videoElement, options = {}) {
        super.initialize();

        this._options = options;
        this._loop = false;
        this._pendingPlaying = false;
        this._isPlaying = false;
        this._endListeners = [];

        this.setVideoElement(videoElement);
    }

    /**
     * @param {Function} listener
     */
    addEndListener(listener) {
        if (this.isEnded() || !this._videoElement) {
            listener(this);
        } else {
            this._endListeners.push(listener);
        }
    }

    applyOptions() {
        if (this._options.loop) {
            this.loop = true;
        }

        if (this._options.muted) {
            this.muted = true;
        }

        if (Number.isFinite(this._options.volume)) {
            this.volume = this._options.volume;
        }

        if (Number.isFinite(this._options.playbackRate)) {
            this.defaultPlaybackRate = this._options.playbackRate;
            this.playbackRate = this._options.playbackRate;
        }

        if (Number.isFinite(this._options.defaultPlaybackRate)) {
            this.defaultPlaybackRate = this._options.defaultPlaybackRate;
        }
    }

    clearEndListeners() {
        this._endListeners = [];
    }

    destroy() {
        this.destroyVideo();
        PIXI.Sprite.prototype.destroy.call(this,
            { children: true, texture: true, baseTexture: true });
    }

    destroyVideo() {
        if (this._videoElement) {
            this._videoElement.destroy();
        }
    }

    endVideo() {
        if (this._videoElement) {
            this.currentTime = this.duration;
            this.pause();
        }
    }

    isEnded() {
        if (!this._videoElement) {
            return false;
        }

        return this.currentTime >= this.duration && this.isReady();
    }

    isPaused() {
        if (!this._videoElement) {
            return false;
        }

        return this.video.paused;
    }

    isPlaying() {
        return this._isPlaying || this._pendingPlaying;
    }

    isReady() {
        return this._videoElement.isReady();
    }

    /**
     * @return {Promise}
     */
    onEnd() {
        return new Promise((resolve) => {
            this.addEndListener(() => {
                resolve(this);
            });
        });
    }

    /**
     * @param {VideoElement | GifElement} videoElement
     */
    setVideoElement(videoElement) {
        if (videoElement) {
            this._videoElement = videoElement;
            this._videoElement.addLoadListener(this._onLoad.bind(this));
        } else if (this._videoElement) {
            this.stop();
            this.destroy();
            this._videoElement = null;
        } else {
            this._videoElement = null;
        }
    }

    /**
     * @param {Number} [currentTime=0]
     * @return {Promise}
     */
    play(currentTime = 0) {
        if (!this._videoElement) {
            return Promise.reject();
        }

        this._pendingPlaying = true;

        if (Utils.RPGMAKER_NAME === 'MV' && VideoPlayerParams.useMasterVolume) {
            this.volume = AudioManager.masterVolume * 100;
        }

        return new Promise((resolve) => {
            this._videoElement.addLoadListener(() => {
                this._videoElement.play(currentTime);
                this._pendingPlaying = false;
                this._isPlaying = true;
                resolve(this);
            });
        });
    }

    resume() {
        return this.play(this.currentTime);
    }

    pause() {
        if (this._videoElement) {
            this._videoElement.pause();
        }

        this._pendingPlaying = false;
        this._isPlaying = false;
    }

    stop() {
        if (this._videoElement) {
            this.pause();
            this.currentTime = 0;
        }

        this._pendingPlaying = false;
        this._isPlaying = false;
    }

    updateEnd() {
        while (this._endListeners.length > 0) {
            this._endListeners.shift()(this);
        }
    }

    update() {
        super.update();

        if (this.visible && this.isReady()) {
            if (this._videoElement instanceof VideoElement) {
                if (Utils.RPGMAKER_NAME === 'MV') {
                    this.texture.update();
                }
            } else if (this._videoElement instanceof GifElement) {
                this._videoElement.updateFrame();
            }
        }
    }

    // private methods

    _onLoad() {
        if (this._videoElement instanceof VideoElement) {
            this.texture = PIXI.Texture.from(this._videoElement.video);
        } else if (this._videoElement instanceof GifElement) {
            this.texture = this._videoElement.video.texture;
        }

        this.width = this._videoElement.width;
        this.height = this._videoElement.height;
        this.currentTime = 0;

        this.applyOptions();

        if (this._options.autoplay) {
            this.play();
        } else {
            this.stop();
        }

        if (this._videoElement instanceof VideoElement) {
            this._videoElement.video.onended = this._onEnd.bind(this);
        } else if (this._videoElement instanceof GifElement) {
            this._videoElement.video.onLoop = this._onEnd.bind(this);
        }
    }

    _onEnd() {
        this.endVideo();
        this.updateEnd();

        if (this.loop) {
            this.play();
        }
    }

    // properties

    /**
     * @return {Boolean}
     */
    get autoplay() {
        return (this.video ?
            this.video.autoplay : false);
    }

    set autoplay(value) {
        if (this.video) {
            this.video.autoplay = value;
        }
    }

    /**
     * @return {Number}
     */
    get currentTime() {
        return (this._videoElement ?
            this._videoElement.currentTime : 0);
    }

    set currentTime(value) {
        if (this.video) {
            this.video.currentTime = value;
        }
    }

    /**
     * @return {Number}
     */
    get defaultPlaybackRate() {
        return (this.video ?
            this.video.defaultPlaybackRate * 100 : 100);
    }

    set defaultPlaybackRate(value) {
        if (this.video) {
            this.video.defaultPlaybackRate = value / 100;
        }
    }

    /**
     * @return {Number}
     */
    get duration() {
        return (this._videoElement ?
            this._videoElement.duration : 0);
    }

    /**
     * @return {Boolean}
     */
    get loop() {
        return this._loop;
    }

    set loop(value) {
        this._loop = value;
    }

    /**
     * @return {Boolean}
     */
    get muted() {
        return (this.video ?
            this.video.muted : true);
    }

    set muted(value) {
        if (this.video) {
            this.video.muted = value;
        }
    }

    set options(value) {
        this._options = value;
    }

    /**
     * @return {Number}
     */
    get playbackRate() {
        return (this.video ?
            this.video.playbackRate * 100 : 100);
    }

    set playbackRate(value) {
        if (this.video) {
            this.video.playbackRate = value / 100;
        }
    }

    /**
     * @return {Number}
     */
    get volume() {
        return (this.video ?
            this.video.volume * 100 : 100);
    }

    set volume(value) {
        if (this.video) {
            this.video.volume = value / 100;
        }
    }

    /**
     * @return {HTMLVideoElement}
     */
    get video() {
        if (this._videoElement instanceof VideoElement) {
            return this._videoElement.video || null;
        }

        return null;
    }

    /**
     * @return {VideoElement | GifElement}
     */
    get videoElement() {
        return this._videoElement;
    }

}

if (Utils.RPGMAKER_NAME === 'MV') {

    Sprite_Video.prototype.show = function() {
        this.visible = true;
    };

    Sprite_Video.prototype.hide = function() {
        this.visible = false;
    };

}

//===========================================================================
// Sprite_VideoPlayer
//===========================================================================

class Sprite_VideoPlayer extends Sprite {

    initialize() {
        super.initialize();

        this._videos = [];
        this._animations = [];
        this._replacements = {};

        this.createLayers();
    }

    createLayers() {
        let maxLayers = 5;

        if (SceneManager._scene instanceof Scene_Map) {
            maxLayers = VideoPlayerParams.mapMaxLayers || 5;
        } else if (SceneManager._scene instanceof Scene_Battle) {
            maxLayers = VideoPlayerParams.battleMaxLayers || 5;
        }

        for (let i = 0; i < maxLayers; i++) {
            const sprite = new Sprite_Video();

            sprite.hide();

            this._videos[i] = sprite;
            this.addChild(sprite);
        }
    }

    clearAnimations(layer) {
        this._animations = this._animations.filter(
            animation => animation.layer !== layer);
    }

    resetLayer(layer) {
        const sprite = this._videos[layer];

        if (sprite) {
            this.setBlendMode(layer, PIXI.BLEND_MODES.NORMAL);
            this.setPivot(layer, 0);
            this.setSpeed(layer, 100);
            this.setVolume(layer, 100);
            this.changeOpacity(layer, 255, 0);
            this.changePos(layer, new Point(0, 0), 0);
            this.changeScale(layer, new Point(1, 1), 0);
        }
    }

    isPlaying(layer) {
        if (layer >= 0) {
            return this._videos[layer] && this._videos[layer].isPlaying();
        }

        return this._videos.some(sprite => sprite && sprite.isPlaying());
    }

    isReplacing(layer) {
        if (layer >= 0) {
            return !!this._replacements[layer];
        }

        return Object.keys(this._replacements).length > 0;
    }

    getVideo(layer) {
        return this._videos[layer] || null;
    }

    setVolume(layer, value) {
        if (this._videos[layer]) {
            this._videos[layer].volume = value;
        }
    }

    setPivot(layer, value) {
        const sprite = this._videos[layer];

        if (Number.isFinite(value)) {
            sprite && sprite.pivot.set(value, value);
        } else {
            if (Utils.RPGMAKER_NAME === 'MV') {
                sprite && sprite.pivot.copy(value);
            } else {
                sprite && sprite.pivot.copyFrom(value);
            }
        }
    }

    setBlendMode(layer, value) {
        if (this._videos[layer]) {
            this._videos[layer].blendMode = value;
        }
    }

    setSpeed(layer, value) {
        if (this._videos[layer]) {
            this._videos[layer].playbackRate = value;
        }
    }

    changePos(layer, pos, duration = 0) {
        const sprite = this._videos[layer];

        if (!sprite) {
            return;
        }

        if (duration > 0) {
            this.addAnimation(layer, 'move', {
                pos,
                offsetX: (pos.x - sprite.x) / duration,
                offsetY: (pos.y - sprite.y) / duration
            }, duration);
        } else {
            if (Utils.RPGMAKER_NAME === 'MV') {
                sprite.position.copy(pos);
            } else {
                sprite.position.copyFrom(pos);
            }
        }
    }

    changeOpacity(layer, opacity, duration = 0) {
        const sprite = this._videos[layer];

        if (!sprite) {
            return;
        }

        if (duration > 0) {
            this.addAnimation(layer, 'opacity', {
                opacity,
                offsetOpacity: (opacity - sprite.opacity) / duration,
            }, duration);
        } else {
            sprite.opacity = opacity;
        }
    }

    changeScale(layer, scale, duration = 0) {
        const sprite = this._videos[layer];

        if (!sprite) {
            return;
        }

        if (duration > 0) {
            this.addAnimation(layer, 'scale', {
                scale,
                offsetX: (scale.x - sprite.scale.x) / duration,
                offsetY: (scale.y - sprite.scale.y) / duration
            }, duration);
        } else {
            if (Utils.RPGMAKER_NAME === 'MV') {
                sprite.scale.copy(scale);
            } else {
                sprite.scale.copyFrom(scale);
            }
        }
    }

    addAnimation(layer, type, data, duration = 1) {
        this._animations.push({ layer, type, data, duration });
    }

    /**
     * @param {Number} layer
     * @param {String} src
     * @param {Object} [options={}]
     * @return {Sprite_Video}
     */
    load(layer, src, options = {}) {
        const sprite = this._videos[layer];

        if (!sprite) {
            throw new Error(`Video on layer ${layer} can not be played!`);
        }

        this.stop(layer);

        sprite.loop = false;
        sprite.options = options;
        sprite.setVideoElement(VideoManager.load(src));

        return sprite;
    }

    /**
     * @param {Number} layer
     * @param {Object} [options={}]
     */
    play(layer, options = {}) {
        const sprite = this._videos[layer];

        if (!sprite) {
            throw new Error(`Video on layer ${layer} is not loaded!`);
        }

        sprite.loop = false;
        sprite.options = options;

        sprite.play().then(() => {
            sprite.show();
        });

        if (options.loop) {
            sprite.loop = true;
        }

        delete this._replacements[layer];
    }

    /**
     * @param {Number} layer
     */
    pause(layer) {
        this._videos[layer] && this._videos[layer].pause();
    }

    /**
     * @param {Number} layer
     */
    resume(layer) {
        this._videos[layer] && this._videos[layer].resume();
    }

    /**
     * @param {Number} layer
     * @param {String} src
     * @param {Object} [options={}]
     */
    replace(layer, src, options = {}) {
        const sprite = this._videos[layer];

        if (!sprite) {
            return;
        }

        this._replacements[layer] = true;

        const loadPromise = VideoManager.loadAsync(src);
        const endPromise = sprite.isPlaying() ?
            sprite.onEnd() : Promise.resolve(sprite);

        Promise.all([loadPromise, endPromise]).then(([videoElement]) => {
            const sprite = this._videos[layer];

            this.stop(layer, false);

            sprite.setVideoElement(videoElement);

            this.play(layer, options);
        });
    }

    /**
     * @param {Number} layer
     * @param {Boolean} [resetLayer=true]
     */
    stop(layer, resetLayer = true) {
        const sprite = this._videos[layer];

        if (sprite) {
            if (resetLayer) {
                this.resetLayer(layer);
            }

            sprite.hide();
            sprite.clearEndListeners();
            sprite.stop();
            sprite.destroyVideo();

            this.clearAnimations(layer);
        }

        delete this._replacements[layer];
    }

    stopAll() {
        for (let i = 0; i < this._videos.length; i++) {
            this.stop(i);
        }
    }

    // update methods

    updateBehavior() {
        if (SceneManager._scene instanceof Scene_Map) {
            const okBehavior = VideoPlayerParams.mapOkBehavior;
            const cancelBehavior = VideoPlayerParams.mapCancelBehavior;
            const skipVideo = okBehavior === 'skip' && (Input.isTriggered('ok') || TouchInput.isTriggered())
                || cancelBehavior === 'skip' && (Input.isTriggered('cancel') || TouchInput.isCancelled());

            if (skipVideo) {
                this._videos.forEach((video, layer) => {
                    if (video && video.loop && video.isPlaying()) {
                        this.stop(layer);
                    }
                });
            }
        }
    }

    updateAnimation(animation) {
        const sprite = this._videos[animation.layer];
        const data = animation.data;

        switch (animation.type) {
            case 'move': {
                if (animation.duration > 0) {
                    sprite.x += data.offsetX;
                    sprite.y += data.offsetY;
                } else {
                    if (Utils.RPGMAKER_NAME === 'MV') {
                        sprite.position.copy(data.pos);
                    } else {
                        sprite.position.copyFrom(data.pos);
                    }
                }
                break;
            }
            case 'opacity': {
                if (animation.duration > 0) {
                    sprite.opacity += data.offsetOpacity;
                } else {
                    sprite.opacity = data.opacity;
                }
                break;
            }
            case 'scale': {
                if (animation.duration > 0) {
                    sprite.scale.x += data.offsetX;
                    sprite.scale.y += data.offsetY;
                } else {
                    if (Utils.RPGMAKER_NAME === 'MV') {
                        sprite.scale.copy(data.scale);
                    } else {
                        sprite.scale.copyFrom(data.scale);
                    }
                }
                break;
            }
        }
    }

    updateAnimations() {
        this._animations = this._animations.filter(a => a.duration > 0 && !!this._videos[a.layer]);
        this._animations.forEach((animation) => {
            animation.duration--;
            this.updateAnimation(animation);
        });
    }

    update() {
        this.visible = this.children.length > 0;

        if (this.visible) {
            super.update();
            this.updateAnimations();
        }
    }

}

if (Utils.RPGMAKER_NAME === 'MV') {

    Sprite_VideoPlayer.prototype.show = function() {
        this.visible = true;
    };

    Sprite_VideoPlayer.prototype.hide = function() {
        this.visible = false;
    };

}

//===========================================================================
// SceneManager
//===========================================================================

const VideoPlayer_SceneManager_goto = SceneManager.goto;
SceneManager.goto = function(sceneClass) {
    if (sceneClass === Scene_Title && !Scene_TitleVideo.showed) {
        sceneClass = Scene_TitleVideo;
    }

    VideoPlayer_SceneManager_goto.call(this, sceneClass);
};

//===========================================================================
// Spriteset_Base
//===========================================================================

const VideoPlayer_Spriteset_Base_createUpperLayer =
    Spriteset_Base.prototype.createUpperLayer;
Spriteset_Base.prototype.createUpperLayer = function() {
    VideoPlayer_Spriteset_Base_createUpperLayer.apply(this, arguments);
    this.createVideoPlayer();
};

Spriteset_Base.prototype.createVideoPlayer = function() {
    this._videoPlayer = new Sprite_VideoPlayer();

    VideoManager.setVideoPlayer(this._videoPlayer);

    this.placeVideoPlayerBeforePictures();
};

Spriteset_Base.prototype.placeVideoPlayerBeforePictures = function() {
    this.addChildAt(this._videoPlayer, this.children.indexOf(this._pictureContainer));
};

Spriteset_Base.prototype.placeVideoPlayerAfterPictures = function() {
    this.addChildAt(this._videoPlayer, this.children.indexOf(this._pictureContainer) + 1);
};

//===========================================================================
// Spriteset_Map
//===========================================================================

const VideoPlayer_Spriteset_Map_createParallax =
    Spriteset_Map.prototype.createParallax;
Spriteset_Map.prototype.createParallax = function() {
    const videoParallax = $gameMap.videoParallaxName();

    if (videoParallax) {
        this._parallax = new Sprite_Video();
        this._parallax.setVideoElement(VideoManager.load(videoParallax));
        this._parallax.loop = true;
        this._parallax.muted = true;
        this._parallax.play();
        this._baseSprite.addChild(this._parallax);
    } else {
        VideoPlayer_Spriteset_Map_createParallax.apply(this, arguments);
    }
};

const VideoPlayer_Spriteset_Map_updateParallax =
    Spriteset_Map.prototype.updateParallax;
Spriteset_Map.prototype.updateParallax = function() {
    if (!$gameMap.isVideoParallax()) {
        VideoPlayer_Spriteset_Map_updateParallax.apply(this, arguments);
    } else if (this._parallax.texture) {
        this._parallax.x = -Math.round($gameMap.parallaxOx() % this._parallax.texture.width);
        this._parallax.y = -Math.round($gameMap.parallaxOy() % this._parallax.texture.height);
    }
};

//===========================================================================
// Scene_Base
//===========================================================================

const VideoPlayer_Scene_Base_isReady =
    Scene_Base.prototype.isReady;
Scene_Base.prototype.isReady = function() {
    return VideoPlayer_Scene_Base_isReady.apply(this, arguments)
        && VideoManager.isReady();
};

const VideoPlayer_Scene_Base_terminate =
    Scene_Base.prototype.terminate;
Scene_Base.prototype.terminate = function() {
    VideoPlayer_Scene_Base_terminate.apply(this, arguments);
    VideoManager.setVideoPlayer(null);
};

//===========================================================================
// Scene_Title
//===========================================================================

const VideoPlayer_Scene_Title_createBackground =
    Scene_Title.prototype.createBackground;
Scene_Title.prototype.createBackground = function() {
    VideoPlayer_Scene_Title_createBackground.apply(this, arguments);

    const params = VideoPlayerParams.titleBackground;

    if (params && params.src) {
        this._videoSprite = new Sprite_Video(VideoManager.load(params.src), {
            playbackRate: params.playbackRate,
            volume: params.volume,
            loop: true
        });
        this._videoSprite.play();

		if (this._backSprite1 && this._backSprite1.parent) {
			this._backSprite1.parent.removeChild(this._backSprite1);
		}

        if (this._backSprite2 && this._backSprite2.parent) {
			this._backSprite2.parent.removeChild(this._backSprite2);
		}

		if (Imported['Galv_CustomTitle']) {
			this._sLayer.addChildAt(this._videoSprite, 0);
		} else {
			this.addChild(this._videoSprite);
		}
    }
};

//===========================================================================
// Scene_Map
//===========================================================================

const VideoPlayer_Scene_Map_isMenuEnabled =
    Scene_Map.prototype.isMenuEnabled;
Scene_Map.prototype.isMenuEnabled = function() {
    return VideoPlayer_Scene_Map_isMenuEnabled.apply(this, arguments)
        && !VideoManager.isPlaying();
};

const VideoPlayer_Scene_Map_update =
    Scene_Map.prototype.update;
Scene_Map.prototype.update = function() {
    VideoPlayer_Scene_Map_update.apply(this, arguments);
    VideoManager.update();
};

//===========================================================================
// Scene_Menu
//===========================================================================

const VideoPlayer_Scene_Menu_createBackground =
    Scene_Menu.prototype.createBackground;
Scene_Menu.prototype.createBackground = function() {
    if (VideoPlayerParams.menuBackground) {
        let index = this.children.length;

        if (this._backgroundSprite) {
            index = this.children.indexOf(this._backgroundSprite);

            this.removeChild(this._backgroundSprite);
        }

        this._videoBackgroundSprite = new Sprite_Video(
            VideoManager.load(VideoPlayerParams.menuBackground));
        this._videoBackgroundSprite.loop = true;
        this._videoBackgroundSprite.play();

        this.addChildAt(this._videoBackgroundSprite, index);
    } else {
        VideoPlayer_Scene_Menu_createBackground.apply(this, arguments);
    }
};

//===========================================================================
// Scene_Battle
//===========================================================================

const VideoPlayer_Scene_Battle_updateStatusWindowVisibility =
    Scene_Battle.prototype.updateStatusWindowVisibility;
Scene_Battle.prototype.updateStatusWindowVisibility = function() {
    VideoPlayer_Scene_Battle_updateStatusWindowVisibility.apply(this, arguments);

    if (VideoPlayerParams.statusWindowAutoHide && $gameTroop.isEventRunning() && VideoManager.isBusy()) {
        this._statusWindow.close();
    }
};

//===========================================================================
// Scene_VideoPlayer
//===========================================================================

class Scene_VideoPlayer extends Scene_Base {

    create() {
        this.createVideoSprites();
    }

    createVideoSprites() {
        this._videoSprites = this.videos.map((data) => {
            const videoElement = VideoManager.load(data.src);
            const sprite = new Sprite_Video(videoElement);

            sprite.hide();
            sprite.addEndListener(this.onVideoEnd.bind(this));

            if (data.muted) {
                sprite.muted = true;
            }

            if (data.playbackRate) {
                sprite.defaultPlaybackRate = data.playbackRate;
                sprite.playbackRate = data.playbackRate;
            }

            if (Number.isFinite(data.volume)) {
                sprite.volume = data.volume;
            }

            this.addChild(sprite);

            return sprite;
        });
    }

    onVideoEnd() {
        this.stopVideo();
        this.playNextVideo();

        if (!this._videoSprite) {
            this.popScene();
        }
    }

    stopVideo() {
        if (this._videoSprite) {
            this._videoSprite.stop();
            this._videoSprite.destroy();
        }
    }

    playNextVideo() {
        this._videoSprite = this._videoSprites.shift() || null;

        if (this._videoSprite) {
            this._videoSprite.play();
            this._videoSprite.show();
        }
    }

    start() {
        super.start();
        this.playNextVideo();
    }

    updateBehavior() {
        const okBehavior = this.okBehavior;
        const cancelBehavior = this.cancelBehavior;
        const skipVideo = okBehavior === 'skip' && (Input.isTriggered('ok') || TouchInput.isTriggered())
            || cancelBehavior === 'skip' && (Input.isTriggered('cancel') || TouchInput.isCancelled());

        if (skipVideo) {
            this.onVideoEnd();
            return;
        }

        const fastForward = okBehavior === 'fastForward' && (Input.isPressed('ok') || TouchInput.isPressed())
            || cancelBehavior === 'fastForward' && (Input.isPressed('cancel') || TouchInput.isPressed());

        if (fastForward) {
            this._videoSprite.playbackRate = this.fastForwardPlaybackRate;
        } else {
            this._videoSprite.playbackRate = this._videoSprite.defaultPlaybackRate;
        }
    }

    update() {
        super.update();

        if (this._videoSprite) {
            this.updateBehavior();
        }
    }

    // properties

    get videos() {
        return [];
    }

    get okBehavior() {
        return null;
    }

    get cancelBehavior() {
        return null;
    }

    get fastForwardPlaybackRate() {
        return 100;
    }

}

//===========================================================================
// Scene_TitleVideo
//===========================================================================

class Scene_TitleVideo extends Scene_VideoPlayer {

    popScene() {
        Scene_TitleVideo.showed = true;
        SceneManager.goto(Scene_Title);
    }

    // properties

    get videos() {
        return VideoPlayerParams.titleVideos.videos;
    }

    get okBehavior() {
        return VideoPlayerParams.titleVideos.okBehavior;
    }

    get cancelBehavior() {
        return VideoPlayerParams.titleVideos.cancelBehavior;
    }

    get fastForwardPlaybackRate() {
        return VideoPlayerParams.titleVideos.playbackRate;
    }

}

Scene_TitleVideo.showed = VideoPlayerParams.titleVideos.videos.length === 0;

if (Imported['DKTools'] && DKToolsParam.get('Quick Start', 'Enabled')) {

    Scene_TitleVideo.showed = true;

}

//===========================================================================
// Scene_CreditsVideo
//===========================================================================

class Scene_CreditsVideo extends Scene_VideoPlayer {

    // properties

    get videos() {
        return VideoPlayerParams.creditsVideos.videos;
    }

    get okBehavior() {
        return VideoPlayerParams.creditsVideos.okBehavior;
    }

    get cancelBehavior() {
        return VideoPlayerParams.creditsVideos.cancelBehavior;
    }

    get fastForwardPlaybackRate() {
        return VideoPlayerParams.creditsVideos.playbackRate;
    }

}

//===========================================================================
// Credits command
//===========================================================================

if (VideoPlayerParams.creditsVideos.commandName) {

    //===========================================================================
    // Scene_Title
    //===========================================================================

    const VideoPlayer_Scene_Title_createCommandWindow =
        Scene_Title.prototype.createCommandWindow;
    Scene_Title.prototype.createCommandWindow = function() {
        VideoPlayer_Scene_Title_createCommandWindow.apply(this, arguments);

        if (this._commandWindow) {
            this._commandWindow.setHandler('credits', this.commandCredits.bind(this));
        }
    };

    Scene_Title.prototype.commandCredits = function() {
        this._commandWindow && this._commandWindow.close();
        SceneManager.push(Scene_CreditsVideo);
    };

    //===========================================================================
    // Window_TitleCommand
    //===========================================================================

    const VideoPlayer_Window_TitleCommand_makeCommandList =
        Window_TitleCommand.prototype.makeCommandList;
    Window_TitleCommand.prototype.makeCommandList = function() {
        VideoPlayer_Window_TitleCommand_makeCommandList.apply(this, arguments);
        this.addCommand(VideoPlayerParams.creditsVideos.commandName, 'credits');
    };

}

//===========================================================================
// Compatibility with other plugins
//===========================================================================

if (Imported['DK_Game_Time']) {

    const VideoPlayer_Game_System_isGameTimeWindowVisible =
        Game_System.prototype.isGameTimeWindowVisible;
    Game_System.prototype.isGameTimeWindowVisible = function() {
        return VideoPlayer_Game_System_isGameTimeWindowVisible.apply(this, arguments)
            && !VideoManager.isPlaying();
    };

}